implementation module StdControlClass


//	Clean Object I/O library, version 1.1

//	Definition of the Controls class for controls.


import	StdBool, StdFunc, StdInt, StdList, StdMisc, StdTuple
import	commondef, controldefaccess, controlvalidate, id, StdControlDef, StdPSt, windowhandle
import	ospicture, ossystem, oswindow


class Controls cdef
where
	controlToHandles:: !(cdef .ls (PSt .l .p))	-> [ControlState .ls (PSt .l .p)]
	getControlType	::  (cdef .ls .ps)			-> ControlType

/*	Translating control elements with local state into the internal representation.
	Note that no additional information is generated yet.
	Attributes that can be placed in the relevant record fields 
		wItemId		- ControlId
		wItemShow	- ControlHide
		wItemSelect	- ControlSelectState
		wItemLook	- ControlLook
		wItemInfo	- ControlDomain
	are removed from the attribute list. 
	The remaining attribute list is copied to wItemAtts.
*/

instance Controls (AddLS c)	| Controls c
where
	controlToHandles :: !(AddLS c .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]	| Controls c
	controlToHandles {addLS,addDef}
		=	[WElementHandleToControlState
				(WExtendLSHandle {	wExtendLS		= addLS
								 ,	wExtendItems	= map ControlStateToWElementHandle (controlToHandles addDef)
								 }
				)
			]
	getControlType _
		=	""

instance Controls (NewLS c)	| Controls c
where
	controlToHandles :: !(NewLS c .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]	| Controls c
	controlToHandles {newLS,newDef}
		=	[WElementHandleToControlState
				(WChangeLSHandle {	wChangeLS		= newLS
								 ,	wChangeItems	= map ControlStateToWElementHandle (controlToHandles newDef)
								 }
				)
			]
	getControlType _
		=	""

instance Controls (ListLS c)	| Controls c
where
	controlToHandles :: !(ListLS c .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]	| Controls c
	controlToHandles (ListLS c)
		=	[WElementHandleToControlState (WListLSHandle (map ControlStateToWElementHandle (flatten (map controlToHandles c))))]
	getControlType _
		=	""

instance Controls NilLS
where
	controlToHandles :: !(NilLS .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles NilLS
		=	[WElementHandleToControlState (WListLSHandle [])]
	getControlType _
		=	""

instance Controls ((:+:) c1 c2)	| Controls c1 & Controls c2
where
	controlToHandles :: !((:+:) c1 c2 .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]	| Controls c1 & Controls c2
	controlToHandles (c1:+:c2)
		=	controlToHandles c1++controlToHandles c2
	getControlType _
		=	""

instance Controls RadioControl
where
	controlToHandles :: !(RadioControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (RadioControl items layout index atts)
		#	(nrItems,items)	= Ulength items
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsRadioControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= RadioInfo 
										{	radioItems = map radioItemToInfo items
										,	radioLayout= validateLayout nrItems layout
										,	radioIndex = SetBetween index 1 nrItems
										}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"RadioControl"

instance Controls CheckControl
where
	controlToHandles :: !(CheckControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (CheckControl items layout atts)
		#	(nrItems,items)	= Ulength items
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsCheckControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= CheckInfo
										{	checkItems = map checkItemToInfo items
										,	checkLayout= validateLayout nrItems layout
										}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"CheckControl"

instance Controls PopUpControl
where
	controlToHandles :: !(PopUpControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (PopUpControl popUpItems index atts)
		#	nrItems	= length popUpItems
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsPopUpControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= PopUpInfo 
										{	popUpInfoItems = popUpItems
										,	popUpInfoIndex = validatePopUpIndex nrItems index
										}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"PopUpControl"

instance Controls SliderControl
where
	controlToHandles :: !(SliderControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (SliderControl direction length sliderState action atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsSliderControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= SliderInfo 
				 						{	sliderInfoDir	= direction
				 						,	sliderInfoLength= length
			 							,	sliderInfoState	= validateSliderState sliderState
			 							,	sliderInfoAction= action
			 							}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"SliderControl"

instance Controls TextControl
where
	controlToHandles :: !(TextControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (TextControl textLine atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsTextControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= TextInfo {textInfoText=textLine}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"TextControl"

instance Controls EditControl
where
	controlToHandles :: !(EditControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (EditControl textLine width nrLines atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsEditControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= EditInfo
										{	editInfoText	= textLine
										,	editInfoWidth	= width
										,	editInfoNrLines	= nrLines
										}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"EditControl"

instance Controls ButtonControl
where
	controlToHandles :: !(ButtonControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (ButtonControl textLine atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsButtonControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= ButtonInfo {buttonInfoText=textLine}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"ButtonControl"

instance Controls CustomButtonControl
where
	controlToHandles :: !(CustomButtonControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (CustomButtonControl size controlLook atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsCustomButtonControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= CustomButtonInfo {cButtonInfoLook={lookFun=validateLook controlLook,lookPen=defaultPen}}
				,	wItemAtts		= [ControlSize size:filter (not o redundantAttribute) atts]
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"CustomButtonControl"

instance Controls CustomControl
where
	controlToHandles :: !(CustomControl .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]
	controlToHandles (CustomControl size controlLook atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsCustomControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= CustomInfo {customInfoLook={lookFun=validateLook controlLook,lookPen=defaultPen}}
				,	wItemAtts		= [ControlSize size:filter (not o redundantAttribute) atts]
				,	wItems			= []
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	getControlType _
		=	"CustomControl"

instance Controls (CompoundControl c)	| Controls c
where
	controlToHandles :: !(CompoundControl c .ls (PSt .l .p)) -> [ControlState .ls (PSt .l .p)]	| Controls c
	controlToHandles (CompoundControl controls atts)
		=	[WElementHandleToControlState
				(WItemHandle 
				{	wItemId			= getIdAttribute atts
				,	wItemNr			= 0
				,	wItemKind		= IsCompoundControl
				,	wItemShow		= not (Contains iscontrolhide atts)
				,	wItemSelect		= getSelectStateAttribute atts
				,	wItemInfo		= CompoundInfo
										{	compoundDomain	= domain
										,	compoundOrigin	= origin
										,	compoundHScroll	= if hasHScroll (Just hScrollInfo) Nothing
										,	compoundVScroll	= if hasVScroll (Just vScrollInfo) Nothing
										,	compoundLookInfo= if hasLook (Just {compoundLook={lookFun=validateLook (getcontrollook lookAtt),lookPen=defaultPen}
																			   ,compoundClip={clipRgn=0,clipOk=False}
																			   }
																		 )
																		 Nothing
										}
				,	wItemAtts		= filter (not o redundantAttribute) atts
				,	wItems			= map ControlStateToWElementHandle (controlToHandles controls)
				,	wItemPos		= zero
				,	wItemFixedPos	= False
				,	wItemSize		= zero
				,	wItemPtr		= OSNoWindowPtr
				})
			]
	where
		(hasHScroll,hScrollAtt)	= Select iscontrolhscroll undef atts
		(hasVScroll,vScrollAtt)	= Select iscontrolvscroll undef atts
		(hasLook,lookAtt)		= Select iscontrollook undef atts
		defaultDomain			= ControlViewDomain {corner1=zero,corner2={x=MaxSigned2ByteInt,y=MaxSigned2ByteInt}}
		(_,domainAtt)			= Select iscontrolviewdomain defaultDomain atts
		domain					= validateCompoundDomain (getcontroldomain domainAtt)
		(_,originAtt)			= Select iscontrolorigin (ControlOrigin domain.corner1) atts
		origin					= validateOrigin domain (getcontrolorigin originAtt)
		hScrollInfo				= {	scrollFunction	= getcontrolhscrollfunction hScrollAtt
								  ,	scrollItemPos	= zero
								  ,	scrollItemSize	= zero
								  ,	scrollItemPtr	= OSNoWindowPtr
								  }
		vScrollInfo				= {	scrollFunction	= getcontrolvscrollfunction vScrollAtt
								  ,	scrollItemPos	= zero
								  ,	scrollItemSize	= zero
								  ,	scrollItemPtr	= OSNoWindowPtr
								  }
	getControlType _
		=	"CompoundControl"


//	Additional functions:

getIdAttribute :: ![ControlAttribute .ps] -> Maybe Id
getIdAttribute atts
	|	hasId
		=	Just (getcontrolid idAtt)
		=	Nothing
where
	(hasId,idAtt)	= Select iscontrolid undef atts

getSelectStateAttribute :: ![ControlAttribute .ps] -> Bool
getSelectStateAttribute atts
	=	enabled (getcontrolselectstate (snd (Select iscontrolselectstate (ControlSelectState Able) atts)))

redundantAttribute :: !(ControlAttribute .ps) -> Bool
redundantAttribute (ControlId _)			= True
redundantAttribute ControlHide				= True
redundantAttribute (ControlSelectState _)	= True
redundantAttribute (ControlLook _)			= True
redundantAttribute (ControlViewDomain _)	= True
redundantAttribute _						= False

validateLayout :: !Int !RowsOrColumns -> RowsOrColumns
validateLayout nrItems (Rows    n) = Rows    (SetBetween n 1 nrItems)
validateLayout nrItems (Columns n) = Columns (SetBetween n 1 nrItems)

validatePopUpIndex :: !Int !Index -> Index
validatePopUpIndex nrItems index
	|	IsBetween index 1 nrItems
		=	index
		=	1

validateCompoundDomain :: !ViewDomain -> ViewDomain
validateCompoundDomain domain
	=	{corner1={x=l,y=t},corner2={x=max r (l+minWidth),y=max b (t+minHeight)}}
where
	(l,t, r,b)			= RectangleToRect domain
	(minWidth,minHeight)= OSMinCompoundSize

validateOrigin :: !ViewDomain !Point -> Point
validateOrigin domain origin
	=	{	x=SetBetween origin.x domain.corner1.x domain.corner2.x
		,	y=SetBetween origin.y domain.corner1.y domain.corner2.y
		}

radioItemToInfo :: !(RadioControlItem .ps) -> RadioItemInfo .ps
radioItemToInfo item
	=	{radioItem=item,radioItemPos=zero,radioItemSize=zero,radioItemPtr=OSNoWindowPtr}

checkItemToInfo :: !(CheckControlItem .ps) -> CheckItemInfo .ps
checkItemToInfo item
	=	{checkItem=item,checkItemPos=zero,checkItemSize=zero,checkItemPtr=OSNoWindowPtr}
